<?php

namespace Modules\LiveChat\Repositories;

use App\Models\Setting;
use App\Models\User;
use App\Traits\ApiReturnFormatTrait;
use App\Traits\CommonHelperTrait;
use App\Traits\FileUploadTrait;
use Illuminate\Support\Facades\DB;
use Modules\LiveChat\Entities\Message;
use Modules\LiveChat\Entities\MessageUser;
use Modules\LiveChat\Interfaces\LiveChatInterface;
use Modules\LiveChat\Traits\PusherTrait;

class LiveChatRepository implements LiveChatInterface
{
    use ApiReturnFormatTrait, FileUploadTrait, PusherTrait, CommonHelperTrait;

    private $messageModel;
    private $userModel;
    private $messageUser;
    private $settingModel;

    public function __construct(
        Message $messageModel,
        User $userModel,
        Setting $settingModel,
        MessageUser $messageUserModel

    ) {
        $this->messageModel = $messageModel;
        $this->userModel = $userModel;
        $this->messageUser = $messageUserModel;
        $this->settingModel = $settingModel;
    }

    public function model()
    {
        return $this->messageModel;
    }

    public function store($request)
    {

        DB::beginTransaction(); // start database transaction
        try {

            $receiver = $this->userModel->find(decryptFunction($request->user_id));
            if (!$receiver) {
                return $this->responseWithError(___('alert.user_not_found'));
            }
            $message = $this->messageModel->create([
                'message' => $request->message,
                'sender_id' => auth()->user()->id,
                'receiver_id' => decryptFunction($request->user_id),
            ]);
            $result = $this->sendMessage($message, $receiver);
            if (!$result['status']) {
                return $this->responseWithError($result['message'], [], 400);
            }
            DB::commit(); // commit database transaction
            return $this->responseWithSuccess(___('alert.message_send'), [], 200);
        } catch (\Throwable $th) {
            DB::rollBack(); // rollback database transaction
            return $this->responseWithError($th->getMessage(), [], 400);
        }

    }

    public function update($request)
    {
        try {
            if ($request->has('pusher_app_id')) {
                $pusher_app_id = $this->settingModel::where('name', 'pusher_app_id')->first();
                if ($pusher_app_id) {
                    $pusher_app_id->update(['value' => $request->pusher_app_id]);
                } else {
                    $this->settingModel::create([
                        'name' => 'pusher_app_id',
                        'value' => $request->pusher_app_id,
                    ]);
                }
                $this->setEnvironmentValue('PUSHER_APP_ID', $request->pusher_app_id);
            }
            if ($request->has('pusher_app_key')) {
                $pusher_app_key = $this->settingModel::where('name', 'pusher_app_key')->first();
                if ($pusher_app_key) {
                    $pusher_app_key->update(['value' => $request->pusher_app_key]);
                } else {
                    $this->settingModel::create([
                        'name' => 'pusher_app_key',
                        'value' => $request->pusher_app_key,
                    ]);
                }
                $this->setEnvironmentValue('PUSHER_APP_KEY', $request->pusher_app_key);
            }
            if ($request->has('pusher_app_secret')) {
                $pusher_app_secret = $this->settingModel::where('name', 'pusher_app_secret')->first();
                if ($pusher_app_secret) {
                    $pusher_app_secret->update(['value' => $request->pusher_app_secret]);
                } else {
                    $this->settingModel::create([
                        'name' => 'pusher_app_secret',
                        'value' => $request->pusher_app_secret,
                    ]);
                }
                $this->setEnvironmentValue('PUSHER_APP_SECRET', $request->pusher_app_secret);
            }
            if ($request->has('pusher_app_cluster')) {
                $pusher_app_cluster = $this->settingModel::where('name', 'pusher_app_cluster')->first();
                if ($pusher_app_cluster) {
                    $pusher_app_cluster->update(['value' => $request->pusher_app_cluster]);
                } else {
                    $this->settingModel::create([
                        'name' => 'pusher_app_cluster',
                        'value' => $request->pusher_app_cluster,
                    ]);
                }
                $this->setEnvironmentValue('PUSHER_APP_CLUSTER', $request->pusher_app_cluster);
            }
            return $this->responseWithSuccess(___('alert.Live_chat_settings_update'), [], 200);
        } catch (\Throwable $th) {
            return $this->responseWithError($th->getMessage(), [], 400);
        }
    }

    public function readMessages($id)
    {
        try {
            $this->model()->where('id', decryptFunction($id))->update(['is_seen' => 1]);
            return $this->responseWithSuccess(___('alert.message_read'), [], 200);
        } catch (\Throwable $th) {
            return $this->responseWithError($th->getMessage(), [], 400);
        }
    }

}
